<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\User\CreateUserRequest;
use App\Models\PreferencePrompt;
use App\Models\SubCategory;
use App\Models\User;
use App\Models\UserPrompt;
use App\Traits\ApiResponder;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    use ApiResponder;

    public function index()
    {
        $users = User::with('roles')->where("state", 1)->orderBy("id")->get();
        return $this->success('Users list.', $users);
    }

    public function listPromptsByUserid($id): JsonResponse
    {
        $prompts = UserPrompt::select([
            'user_prompts.id',
            'user_prompts.sub_category_id',
            'sub_categories.name as sub_category_name',
            'prompts.prompt_input as prompt_input',
            'prompts.prompt_output as prompt_output',
        ])
            ->leftJoin('sub_categories', 'user_prompts.sub_category_id', '=', 'sub_categories.id')
            ->leftJoin('prompts', 'user_prompts.prompt_id', '=', 'prompts.id')
            ->where('user_prompts.state', 1)
            ->orderBy('user_prompts.id')
            ->get();

        return $this->success('Prompts list.', $prompts);
    }

    public function store(CreateUserRequest $request)
    {
        DB::beginTransaction();

        try {
            $user = User::create([
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'username' => $request->username,
                'password' => Hash::make($request->password),
                'email' => $request->email
            ]);

            $role = Role::findById($request->role_id);
            $user->assignRole($role);

            $subCategories = SubCategory::where("state", 1)->orderBy("id")->get();

            foreach ($subCategories as $subCategory) {
                PreferencePrompt::create([
                    'user_id' => $user->id,
                    'sub_category_id' => $subCategory->id,
                    'text' => 0
                ]);
            }

            foreach ($subCategories as $subCategory) {
                UserPrompt::create([
                    'user_id' => $user->id,
                    'sub_category_id' => $subCategory->id
                ]);
            }

            DB::commit();

            return $this->success('User created successfully.', $user);

        } catch (Exception $ex) {
            DB::rollBack();
            return $this->error('Error creating User.', 500, $ex);
        }
    }

    public function show($id)
    {
        $user = User::with('roles')->find($id);

        if (!$user) {
            return $this->error('User not found.');
        }

        return $this->success('User.', $user);
    }

    public function destroy( $id )
    {
        $user = User::find($id);

        if (!$user) {
            return $this->error('User not found.');
        }

        $user->update(['state' => 0]);

        return $this->success("User deleted successfully.");
    }

    public function update(Request $request, string $id): JsonResponse
    {
        try {
            $user = User::find($id);

            $data = $request->except('password');

            if ($request->filled('password')) {
                $data['password'] = Hash::make($request->password);
            }

            $user->update($data);

            return $this->success('User updated successfully.', $user);

        } catch (Exception $ex) {
            return $this->error('Error updating User.', 500, $ex);
        }
    }

    public function addRole(Request $request, string $id): JsonResponse
    {
        try {
            $user = User::find($id);
            $role = Role::findById($request->role_id);

            if($role) {
                $user->assignRole($role);
            } else {
                return $this->error('Role not found.');
            }

            return $this->success('Role added to user successfully.');

        } catch (Exception $ex) {
            return $this->error('Error updating User.', 500, $ex);
        }
    }

    public function removeRole(Request $request, string $id): JsonResponse
    {
        try {
            $user = User::find($id);
            $role = Role::findById($request->role_id);

            if ($role) {
                $user->removeRole($role);
                return $this->success('Role removed from user successfully.');
            } else {
                return $this->error('Role not found.');
            }

        } catch (Exception $ex) {
            return $this->error('Error removing role from user.', 500, $ex);
        }
    }

    public function getRoles(Request $request, string $id)
    {
        try {
            $user = User::find($id);
            $roles = $user->roles()->pluck('name')->all();

            $data = [
                'user' => $user,
                'roles' => $roles
            ];

            $jsonData = json_encode($data);

            return $this->success('Roles of the user:', $jsonData);

        } catch (Exception $ex) {
            return $this->error('Error listing user roles.', 500, $ex);
        }
    }

    public function updatePrompt($id, Request $request): JsonResponse
    {
        try {
            $userPrompt = UserPrompt::where('user_id', $request->$id)
                                    ->where('sub_category_id', $request->sub_category_id)
                                    ->firstOrFail();

            $userPrompt->update([
                'prompt_id' => $request->prompt_id
            ]);

            return $this->success('User prompt updated successfully.');

        } catch (Exception $ex) {
            return $this->error('Error updating User prompt.', 500, $ex);
        }
    }
}
