<?php

namespace App\Http\Controllers\Api;

use App\Http\Requests\CreatePromptRequest;
use App\Http\Requests\ExecutePromptRequest;
use App\Http\Requests\UpdatePromptRequest;
use App\Models\DayOption;
use App\Models\Detail;

use App\Models\Prompt;
use App\Models\PromptHistory;
use App\Models\Resource;
use App\Models\Type;

use App\Traits\ApiResponder;
use App\Traits\ChatGpt;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;

class PromptController extends Controller
{
    use ApiResponder;
    use ChatGpt;


    public function execute(ExecutePromptRequest $request): JsonResponse
    {
        try {
            $prompt = $request->prompt;
            $response = $this->getChatGptResponse($prompt);

            return $this->success('Response.', json_encode($response));

        } catch (Exception $e) {
            return $this->error('Error occurred', 500, $e);
        }
    }

    public function store(CreatePromptRequest $request): JsonResponse
    {
        $nroElements = 5;

        try {
            DB::beginTransaction();

            $user = $request->user();
            $isText = $request->is_text;

            if ($isText == 0) $this->saveWithOptions($user, $request, $nroElements);
            else $this->saveOnlyText($user, $request);

            DB::commit();

            return $this->success('Prompt created successfully.');

        } catch (Exception $ex) {
            DB::rollBack();
            return $this->error('Error creating Prompt.', 500, $ex);
        }
    }

    public function updateScore(UpdatePromptRequest $request, $id)
    {
        $newScore = $request->score;
        $prompt = Prompt::find($id);
        $prompt->update([
            'score' => $newScore
        ]);

        return $this->success('Prompt updated successfully.');
    }

    public function listHistory()
    {
        $historyPrompts = PromptHistory::with([
            'user',
            'prompt',
            'prompt.subCategory',
            'prompt.whereOption',
            'prompt.whenOption',
            'prompt.dayOption',
            'prompt.types',
            'prompt.resources',
            'prompt.details',
            'prompt.language',
            'prompt.subCategory.category'
        ])->get();


        return $this->success('Prompts list.', $historyPrompts);
    }

    public function view()
    {
        $prompts = Prompt::select([
            'prompts.*',
            'users.*'
        ])
            ->leftJoin('categories', 'prompts.category_id', '=', 'categories.id')
            ->leftJoin('sub_categories', 'prompts.sub_category_id', '=', 'sub_categories.id')
            ->where('prompts.state', 1)
            ->orderBy('id')
            ->get();

        return $this->success('Prompts list.', $prompts);
    }

    private function saveWithOptions($user, CreatePromptRequest $request, int $nroElements): void
    {
        $userId = $user->id;

        $types = $request->data['types'];
        $details = $request->data['details'];
        $resources = $request->data['resources'];

        if ($request->day_value == "") $day_value = DayOption::find($request->data['day_id'])->name;
        else $day_value = $request->data['day_value'];

        $prompt = Prompt::create([
            'where_option_id' => $request->data['where_id'],
            'when_option_id' => $request->data['when_id'],
            'day_option_id' => $request->data['day_id'],
            'sub_category_id' => $request->data['sub_category_id'],
            'language_id' => $request->data['language_id'],
            'where_value' => $request->data['where_value'],
            'day_value' => $day_value,
            "prompt_input" => $request->data['prompt_input'],
            "prompt_output" => $request->data['prompt_output'],
            "score" => null,
            "isText" => 0
        ]);

        PromptHistory::create([
            'user_id' => $user->id,
            'prompt_id' => $prompt->id,
            'date' => Carbon::now()
        ]);

        foreach ($types as $type) {
            Type::create([
                'user_id' => $userId,
                'prompt_id' => $prompt->id,
                'description' => $type
            ]);
        }

        foreach ($resources as $resource) {
            Resource::create([
                'user_id' => $userId,
                'prompt_id' => $prompt->id,
                'description' => $resource
            ]);
        }

        foreach ($details as $detail) {
            Detail::create([
                'user_id' => $userId,
                'prompt_id' => $prompt->id,
                'description' => $detail
            ]);
        }

    }

    private function saveOnlyText($user, $request)
    {
        $prompt = Prompt::create([
            'prompt_input' => $request->data['prompt_input'],
            'prompt_output' => $request->data['prompt_output'],
            'sub_category_id' => $request->data['sub_category_id'],
            'type' => $request->data['type'],
            'isText' => 1
        ]);

        PromptHistory::create([
            'user_id' => $user->id,
            'prompt_id' => $prompt->id,
            'date' => Carbon::now()
        ]);
    }
}
